const multer = require('multer');
const path = require('path');
const config = require('../config');
const db = require('../models');
const excelService = require('../services/excelService');
const clientService = require('../services/clientService');
const { Op } = require('sequelize');

const storage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, config.uploadExcelPath),
  filename: (req, file, cb) => cb(null, Date.now() + '-audit-' + (file.originalname || 'upload.xlsx'))
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const ext = path.extname(file.originalname || '').toLowerCase();
    if (['.xlsx', '.xls'].includes(ext)) cb(null, true);
    else cb(new Error('Only Excel files allowed'));
  }
}).single('userfile');

const handleUpload = (req, res, next) => {
  upload(req, res, (err) => {
    if (err) return res.status(400).json({ success: false, message: err.message });
    next();
  });
};

exports.index = async (req, res) => {
  try {
    const employees = await db.Client.findAll({
      where: { user_id: req.userId },
      raw: true
    });
    res.json({ success: true, employees });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.clients = async (req, res) => {
  try {
    const employees = await clientService.getCRCWithBranch(req.userId);
    res.json({ success: true, employees });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.add = async (req, res) => {
  try {
    const [branches, sheets] = await Promise.all([
      db.Branch.findAll({ where: { user_id: req.userId }, raw: true }),
      db.TblSaveAudit.findAll({ order: [['id', 'DESC']], raw: true })
    ]);
    res.json({ success: true, branches, sheets });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.edit = async (req, res) => {
  try {
    const [branches, sheets, client] = await Promise.all([
      db.Branch.findAll({ where: { user_id: req.userId }, raw: true }),
      db.TblSaveAudit.findAll({ order: [['id', 'DESC']], raw: true }),
      db.Client.findByPk(req.params.id)
    ]);
    if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
    res.json({ success: true, branches, sheets, client: client.toJSON() });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.addClient = async (req, res) => {
  try {
    const { ename, epass, ephone, branch_id, refid } = req.body;
    const bcrypt = require('bcrypt');
    const hashedPassword = await bcrypt.hash(epass || '1234', 10);
    const client = await db.Client.create({
      ename,
      epassword: hashedPassword,
      ephone: ephone || null,
      branch_id: branch_id || null,
      refid: refid || null,
      app: 'CRC',
      status: 'offline',
      user_id: req.userId
    });
    res.json({ success: true, client: client.toJSON() });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.editClient = async (req, res) => {
  try {
    const client = await db.Client.findByPk(req.params.id);
    if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
    const { ename, epass, ephone, branch_id, refid } = req.body;
    const updateData = { ename, ephone: ephone || null, branch_id: branch_id || null, refid: refid || null, status: 'offline', user_id: req.userId };
    if (epass) {
      const bcrypt = require('bcrypt');
      updateData.epassword = await bcrypt.hash(epass, 10);
    }
    await client.update(updateData);
    res.json({ success: true, client: client.toJSON() });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.del = async (req, res) => {
  try {
    const client = await db.Client.findByPk(req.params.id);
    if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
    await client.destroy();
    res.json({ success: true, message: 'Deleted' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.display = async (req, res) => {
  try {
    const { refid, client_id, acc_no, page = 1, per_page = 30 } = req.query;
    const where = {};
    if (refid) where.refid = refid;
    if (client_id) where.client_id = client_id;
    if (acc_no) where.k = acc_no;

    const { count, rows } = await db.TblCrcLog.findAndCountAll({
      where,
      include: [{ model: db.Client, as: 'Client', attributes: ['ename'], required: false }],
      limit: parseInt(per_page),
      offset: (parseInt(page) - 1) * parseInt(per_page),
      order: [['id', 'DESC']]
    });

    const [branches, accounts, sheets, clients] = await Promise.all([
      db.TblSaveAudit.findAll({ order: [['id', 'DESC']], raw: true }),
      db.sequelize.query("SELECT k FROM tbl_crc_logs GROUP BY k HAVING COUNT(*)>=2", { type: db.sequelize.QueryTypes.SELECT }),
      db.TblSaveAudit.findAll({ raw: true }),
      db.Client.findAll({ where: { app: 'CRC' }, raw: true })
    ]);

    res.json({
      success: true,
      rows: rows.map(r => ({ ...r.toJSON(), username: r.Client?.ename })),
      branches,
      accounts,
      sheets,
      clients,
      total_rows: count,
      page: parseInt(page),
      per_page: parseInt(per_page)
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.save = [
  handleUpload,
  async (req, res) => {
    try {
      if (!req.file) return res.status(400).json({ success: false, message: 'No file uploaded' });
      const name = req.body.name || 'CRC Import';
      const result = await excelService.importCRC(req.file.path, req.userId, name);
      res.json({ success: true, ...result });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: err.message || 'Import failed' });
    }
  }
];

exports.createExcel = async (req, res) => {
  try {
    const XLSX = require('xlsx');
    const { refid } = req.body;
    const where = refid ? { refid } : {};
    const data = await db.TblAuditExcel.findAll({ where, raw: true });
    const headers = ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p'];
    const rows = data.map(r => headers.map(h => r[h] || ''));
    const ws = XLSX.utils.aoa_to_sheet([headers, ...rows]);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Export');
    const buffer = XLSX.write(wb, { type: 'buffer', bookType: 'xlsx' });
    res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    res.setHeader('Content-Disposition', `attachment; filename="crc-export-${Date.now()}.xlsx"`);
    res.send(buffer);
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Export failed' });
  }
};

exports.delSheet = async (req, res) => {
  try {
    const { refid } = req.body;
    if (!refid) return res.status(400).json({ success: false, message: 'refid required' });
    await db.TblSaveAudit.destroy({ where: { refid } });
    await db.TblCrcLog.destroy({ where: { refid } });
    await db.TblAuditExcel.destroy({ where: { refid } });
    await db.Client.update({ refid: null }, { where: { refid } });
    res.json({ success: true, message: 'Deleted' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Delete failed' });
  }
};
