const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const { body, validationResult } = require('express-validator');
const config = require('../config');
const db = require('../models');

const generateToken = (userId, type) => {
  return jwt.sign(
    { userId, type },
    config.jwtSecret,
    { expiresIn: config.jwtExpiry }
  );
};

const sanitizeUser = (user) => {
  const u = user.toJSON ? user.toJSON() : user;
  delete u.password;
  delete u.epassword;
  return u;
};

exports.adminLogin = [
  body('email').isEmail().normalizeEmail(),
  body('password').notEmpty(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }

      const { email, password } = req.body;
      const user = await db.User.findOne({ where: { email } });

      if (!user) {
        return res.status(401).json({ success: false, message: 'Invalid email or password' });
      }

      const valid = await bcrypt.compare(password, user.password);
      if (!valid) {
        return res.status(401).json({ success: false, message: 'Invalid email or password' });
      }

      const token = generateToken(user.id, 'admin');
      res.json({
        success: true,
        token,
        user: sanitizeUser(user)
      });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Login failed' });
    }
  }
];

exports.clientLogin = [
  body('username').notEmpty().trim(),
  body('password').notEmpty(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }

      const { username, password } = req.body;
      const client = await db.Client.findOne({
        where: { ename: username },
        include: [{ model: db.Branch, as: 'Branch', attributes: ['id', 'name'] }]
      });

      if (!client) {
        return res.status(401).json({ success: false, message: 'Invalid username or password' });
      }

      const valid = await bcrypt.compare(password, client.epassword).catch(() => false);
      if (!valid) {
        const plainMatch = client.epassword === password;
        if (!plainMatch) {
          return res.status(401).json({ success: false, message: 'Invalid username or password' });
        }
      }

      const token = generateToken(client.id, 'client');
      res.json({
        success: true,
        token,
        user: sanitizeUser(client)
      });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Login failed' });
    }
  }
];

exports.me = async (req, res) => {
  try {
    if (req.userType === 'admin') {
      const user = await db.User.findByPk(req.userId);
      if (!user) return res.status(404).json({ success: false, message: 'User not found' });
      return res.json({ success: true, user: sanitizeUser(user), type: 'admin' });
    }
    if (req.userType === 'client') {
      const client = await db.Client.findByPk(req.userId, {
        include: [{ model: db.Branch, as: 'Branch', attributes: ['name'] }]
      });
      if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
      const data = sanitizeUser(client);
      if (client.Branch) data.branch_name = client.Branch.name;
      return res.json({ success: true, user: data, type: 'client' });
    }
    res.status(400).json({ success: false, message: 'Invalid token type' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to get user' });
  }
};
