const { body, validationResult } = require('express-validator');
const db = require('../models');

exports.list = async (req, res) => {
  try {
    const branches = await db.Branch.findAll({
      where: { user_id: req.userId },
      order: [['id', 'DESC']],
      raw: true
    });
    res.json({ success: true, branches });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch branches' });
  }
};

exports.getBranch = async (req, res) => {
  try {
    const branch = await db.Branch.findByPk(req.params.id);
    if (!branch) return res.status(404).json({ success: false, message: 'Branch not found' });
    if (branch.user_id !== req.userId) {
      return res.status(403).json({ success: false, message: 'Forbidden' });
    }
    res.json({ success: true, branch: branch.toJSON() });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch branch' });
  }
};

exports.add = [
  body('bname').notEmpty().trim(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }
      const branch = await db.Branch.create({
        name: req.body.bname,
        user_id: req.userId
      });
      res.json({ success: true, branch: branch.toJSON() });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Failed to add branch' });
    }
  }
];

exports.edit = [
  body('bname').notEmpty().trim(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }
      const branch = await db.Branch.findByPk(req.params.id);
      if (!branch) return res.status(404).json({ success: false, message: 'Branch not found' });
      if (branch.user_id !== req.userId) {
        return res.status(403).json({ success: false, message: 'Forbidden' });
      }
      await branch.update({ name: req.body.bname, user_id: req.userId });
      res.json({ success: true, branch: branch.toJSON() });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Failed to update branch' });
    }
  }
];

exports.del = async (req, res) => {
  try {
    const branch = await db.Branch.findByPk(req.params.id);
    if (!branch) return res.status(404).json({ success: false, message: 'Branch not found' });
    if (branch.user_id !== req.userId) {
      return res.status(403).json({ success: false, message: 'Forbidden' });
    }
    await branch.destroy();
    res.json({ success: true, message: 'Deleted' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to delete' });
  }
};
