const bcrypt = require('bcrypt');
const { body, validationResult } = require('express-validator');
const db = require('../models');
const clientService = require('../services/clientService');

exports.index = async (req, res) => {
  res.json({ message: 'Dashboard' });
};

exports.clients = async (req, res) => {
  try {
    const employees = await clientService.getRESTAFFWithStats(req.userId);
    res.json({ success: true, employees });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch employees' });
  }
};

exports.form = async (req, res) => {
  try {
    const [clients, branches] = await Promise.all([
      db.Client.findAll({ where: { user_id: req.userId }, raw: true }),
      db.Branch.findAll({ where: { user_id: req.userId }, raw: true })
    ]);
    res.json({ success: true, clients, branches });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch data' });
  }
};

exports.getClient = async (req, res) => {
  try {
    const client = await db.Client.findByPk(req.params.id);
    if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
    res.json({ success: true, client: client.toJSON() });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to fetch client' });
  }
};

exports.add = [
  body('ename').notEmpty().trim(),
  body('epass').notEmpty(),
  body('branch_id').optional().isInt(),
  body('refid').optional(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }
      const { ename, epass, ephone, branch_id, refid } = req.body;
      const hashedPassword = await bcrypt.hash(epass, 10);
      const client = await db.Client.create({
        ename,
        epassword: hashedPassword,
        ephone: ephone || null,
        branch_id: branch_id || null,
        refid: refid || null,
        app: 'RESTAFF',
        status: 'offline',
        user_id: req.userId
      });
      res.json({ success: true, client: client.toJSON() });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Failed to add employee' });
    }
  }
];

exports.edit = [
  body('ename').notEmpty().trim(),
  body('epass').optional(),
  body('branch_id').optional().isInt(),
  body('refid').optional(),
  async (req, res) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({ success: false, errors: errors.array() });
      }
      const client = await db.Client.findByPk(req.params.id);
      if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
      if (client.user_id !== req.userId) {
        return res.status(403).json({ success: false, message: 'Forbidden' });
      }

      const { ename, epass, ephone, branch_id, refid } = req.body;
      const updateData = {
        ename,
        ephone: ephone || null,
        branch_id: branch_id || null,
        refid: refid || null,
        status: 'offline',
        user_id: req.userId
      };
      if (epass) {
        updateData.epassword = await bcrypt.hash(epass, 10);
      }
      await client.update(updateData);
      res.json({ success: true, client: client.toJSON() });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: 'Failed to update employee' });
    }
  }
];

exports.del = async (req, res) => {
  try {
    const client = await db.Client.findByPk(req.params.id);
    if (!client) return res.status(404).json({ success: false, message: 'Client not found' });
    if (client.user_id !== req.userId) {
      return res.status(403).json({ success: false, message: 'Forbidden' });
    }
    await client.destroy();
    res.json({ success: true, message: 'Deleted' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed to delete' });
  }
};
