const multer = require('multer');
const path = require('path');
const config = require('../config');
const db = require('../models');
const excelService = require('../services/excelService');
const { Op } = require('sequelize');

const storage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, config.uploadExcelPath),
  filename: (req, file, cb) => cb(null, Date.now() + '-setting-' + (file.originalname || 'upload.xlsx'))
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 },
  fileFilter: (req, file, cb) => {
    const ext = path.extname(file.originalname || '').toLowerCase();
    if (['.xlsx', '.xls'].includes(ext)) cb(null, true);
    else cb(new Error('Only Excel files allowed'));
  }
}).single('userfile');

const handleUpload = (req, res, next) => {
  upload(req, res, (err) => {
    if (err) return res.status(400).json({ success: false, message: err.message });
    next();
  });
};

exports.index = async (req, res) => {
  try {
    await db.SettingExcel.destroy({ where: { user_id: req.userId } });
    res.json({ success: true, message: 'Ready for upload' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.display = async (req, res) => {
  try {
    const { head, page = 1, per_page = 30 } = req.query;
    const where = { user_id: req.userId };
    if (head) {
      const heads = Array.isArray(head) ? head : [head];
      where.header_name = { [Op.in]: heads };
    }

    const { count, rows } = await db.SettingExcel.findAndCountAll({
      where,
      limit: parseInt(per_page),
      offset: (parseInt(page) - 1) * parseInt(per_page),
      raw: true
    });

    const distinct = await db.SettingExcel.findOne({
      where: { user_id: req.userId, f: { [Op.ne]: null }, a: { [Op.notIn]: ['Sr #', 'Print Date :'] } },
      attributes: [[db.sequelize.fn('GROUP_CONCAT', db.sequelize.fn('DISTINCT', db.sequelize.col('header_name')), '@@@'), 'header']],
      raw: true
    });
    const header = distinct?.header ? distinct.header.split('@@@').filter(Boolean) : [];

    res.json({
      success: true,
      excel: rows,
      header,
      total_rows: Math.max(0, count - 4),
      page: parseInt(page),
      per_page: parseInt(per_page)
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.save = [
  handleUpload,
  async (req, res) => {
    try {
      if (!req.file) return res.status(400).json({ success: false, message: 'No file uploaded' });
      const { company_name, add_col_name } = req.body;
      const result = await excelService.importSetting(req.file.path, req.userId, company_name, add_col_name);
      res.json({ success: true, ...result });
    } catch (err) {
      console.error(err);
      res.status(500).json({ success: false, message: err.message || 'Import failed' });
    }
  }
];

exports.summaryBox = async (req, res) => {
  try {
    const company = await db.SettingExcel.findOne({
      where: { user_id: req.userId },
      attributes: ['company_name', 'add_col_name'],
      raw: true
    });
    const remains = await db.sequelize.query(
      'SELECT header_name, COUNT(b) AS raccount, SUM(CAST(p AS DECIMAL(15,2))) AS ramount FROM setting_excel WHERE CAST(p AS DECIMAL) < 0 AND user_id = :uid GROUP BY header_name',
      { replacements: { uid: req.userId }, type: db.sequelize.QueryTypes.SELECT }
    ).catch(() => []);
    const partials = await db.sequelize.query(
      'SELECT header_name, COUNT(l) AS pi FROM setting_excel WHERE CAST(l AS DECIMAL) > 0 AND CAST(p AS DECIMAL) < 0 AND user_id = :uid GROUP BY header_name',
      { replacements: { uid: req.userId }, type: db.sequelize.QueryTypes.SELECT }
    ).catch(() => []);

    res.json({
      success: true,
      company: company?.company_name || 'Rehman Electronics',
      remains,
      partials
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};

exports.excelPdf = async (req, res) => {
  try {
    const { company, additional_coulmn, rib, testheader } = req.method === 'GET' ? req.query : req.body;
    const testheaderArr = testheader ? (Array.isArray(testheader) ? testheader : (typeof testheader === 'string' ? testheader.split(',').map(s => s.trim()).filter(Boolean) : [])) : [];
    const userId = req.userId;

    if (company || additional_coulmn) {
      await db.SettingExcel.update(
        { company_name: company || undefined, add_col_name: additional_coulmn || undefined },
        { where: { user_id: userId } }
      );
    }

    const where = { user_id: userId };
    if (testheaderArr.length) {
      where.header_name = { [Op.in]: testheaderArr };
    }

    const [excel, total_rows, head] = await Promise.all([
      db.SettingExcel.findAll({ where, raw: true }),
      db.SettingExcel.count({ where }),
      db.SettingExcel.findOne({ where: { user_id: userId, a: 'Sr #' }, attributes: ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p'], raw: true })
    ]);

    const companyRow = await db.SettingExcel.findOne({ where: { user_id: userId }, attributes: ['company_name', 'add_col_name'], raw: true });

    res.json({
      success: true,
      company: companyRow?.company_name || 'Rehman Electronics',
      col_name: companyRow?.add_col_name || 'Status',
      rib_col: rib || '',
      total_rows,
      head: head || {},
      excel
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({ success: false, message: 'Failed' });
  }
};
