const XLSX = require('xlsx');
const db = require('../models');

const RESTAFF_HEADERS = ['Sr#', 'Acc#', 'Acc Date', 'Code', 'Customer Name', 'Customer', 'Advance', 'Inquiry Officer', 'Cell#', 'Item', 'PKG Price', 'Remaining Balance', 'Monthly Installment', 'RE.OS', 'Paid', 'UN-Paid RE.OS', 'Last Inst. Date', 'P.Installments', 'Last Remarks'];

const sanitize = (val) => (val != null && val !== undefined ? String(val).trim() : '');

function findHeaderRow(sheet, headers) {
  const range = XLSX.utils.decode_range(sheet['!ref'] || 'A1');
  for (let R = 0; R <= range.e.r; R++) {
    let match = 0;
    const keys = {};
    for (let C = 0; C <= range.e.c; C++) {
      const cell = sheet[XLSX.utils.encode_cell({ r: R, c: C })];
      const val = sanitize(cell?.v ?? '');
      if (headers.includes(val)) {
        keys[val.replace(/\+/g, '')] = C;
        match++;
      }
    }
    if (match >= headers.length - 2) return { row: R, keys };
  }
  return null;
}

exports.importRESTAFF = async (filePath, userId, name, options = {}) => {
  const workbook = XLSX.readFile(filePath);
  const sheet = workbook.Sheets[workbook.SheetNames[0]];
  const data = XLSX.utils.sheet_to_json(sheet, { header: 1, defval: '' });

  const headerInfo = findHeaderRow(sheet, RESTAFF_HEADERS);
  if (!headerInfo) {
    throw new Error('Could not find required headers in Excel file');
  }

  const { keys, row } = headerInfo;
  const refid = Math.floor(Math.random() * 10000000) + 1;
  const startRow = options.startRow || row + 2;
  const endRow = options.endRow || data.length;
  const fixedHeaderName = options.fixedHeaderName;
  let headerName = fixedHeaderName || name || '';

  const records = [];
  for (let i = startRow; i < endRow; i++) {
    const rowData = data[i] || [];
    const getCol = (key) => {
      const idx = keys[key];
      return idx !== undefined ? sanitize(rowData[idx]) : '';
    };

    const col1 = getCol('Sr#');
    const col6 = getCol('Customer');
    if (!col6) continue;

    if (!fixedHeaderName) {
      const isNumeric = /^\d+$/.test(col1);
      if (!isNumeric && col1) {
        headerName = col1;
      }
    }

    const record = {
      header_name: headerName,
      sr_no: getCol('Sr#'),
      acc_no: getCol('Acc#'),
      acc_date: getCol('Acc Date'),
      code: getCol('Code'),
      customer_name: getCol('Customer Name'),
      customer: col6,
      advance: getCol('Advance'),
      inquiry_office: getCol('Inquiry Officer'),
      cell_no: getCol('Cell#'),
      item_no: getCol('Item'),
      price: getCol('PKG Price'),
      balance: getCol('Remaining Balance'),
      installment: getCol('Monthly Installment'),
      os: getCol('RE.OS'),
      paid: getCol('Paid'),
      unpaid: getCol('UN-Paid RE.OS'),
      last_installment_date: getCol('Last Inst. Date'),
      p_installments: getCol('P.Installments'),
      last_remarks: getCol('Last Remarks'),
      refid,
      user_id: userId
    };
    records.push(record);
  }

  if (records.length === 0) {
    throw new Error('No valid records found in Excel file');
  }

  await db.Import.bulkCreate(records);
  await db.TblSave.create({
    name: name || 'Import',
    user_id: userId,
    excel: filePath,
    refid
  });

  return { refid, count: records.length };
};

exports.exportRESTAFF = async (refid = null, branches = null) => {
  let data;
  if (refid) {
    data = await db.Import.findAll({ where: { refid }, raw: true });
  } else if (branches && Array.isArray(branches) && branches.length > 0) {
    data = await db.Import.findAll({
      where: { refid: branches },
      raw: true
    });
  } else {
    data = await db.Import.findAll({ raw: true });
  }

  const headers = ['Sr#', 'Acc#', 'Acc Date', 'Code', 'Customer Name', 'Customer', 'Advance', 'Inquiry Officer', 'Cell#', 'Item', 'PKG Price', 'Remaining Balance', 'Monthly Installment', 'RE.OS', 'Paid', 'UN-Paid RE.OS', 'Last Inst. Date', 'P.Installments', 'Last Remarks', 'Remarks', "CRC's Action", 'Call Date & Time', 'Head Account', 'Ref #'];
  const rows = data.map(r => [
    r.sr_no, r.acc_no, r.acc_date, r.code, r.customer_name, r.customer, r.advance, r.inquiry_office, r.cell_no,
    r.item_no, r.price, r.balance, r.installment, r.os, r.paid, r.unpaid, r.last_installment_date, r.p_installments,
    r.last_remarks, r.remarks, r.cr_action, r.call_time, r.header_name, r.refid
  ]);

  const ws = XLSX.utils.aoa_to_sheet([headers, ...rows]);
  const wb = XLSX.utils.book_new();
  XLSX.utils.book_append_sheet(wb, ws, 'Export');
  return XLSX.write(wb, { type: 'buffer', bookType: 'xlsx' });
};

exports.importCRC = async (filePath, userId, name) => {
  const workbook = XLSX.readFile(filePath);
  const sheet = workbook.Sheets[workbook.SheetNames[0]];
  const data = XLSX.utils.sheet_to_json(sheet, { header: 1, defval: '' });

  const refid = Math.floor(Math.random() * 10000000) + 1;
  const records = [];
  const cols = 'abcdefghijklmnop'.split('');

  for (let i = 0; i < data.length; i++) {
    const row = data[i] || [];
    const obj = {
      header_name: name || '',
      user_id: userId,
      refid
    };
    cols.forEach((c, idx) => {
      obj[c] = sanitize(row[idx]) || null;
    });
    records.push(obj);
  }

  if (records.length === 0) throw new Error('No data in Excel file');

  await db.TblAuditExcel.bulkCreate(records);
  await db.TblSaveAudit.create({
    name: name || 'CRC Import',
    user_id: userId,
    excel: filePath,
    refid
  });

  return { refid, count: records.length };
};

exports.importSetting = async (filePath, userId, companyName, addColName) => {
  const workbook = XLSX.readFile(filePath);
  const sheet = workbook.Sheets[workbook.SheetNames[0]];
  const data = XLSX.utils.sheet_to_json(sheet, { header: 1, defval: '' });

  const records = [];
  const cols = 'abcdefghijklmnop'.split('');

  for (let i = 0; i < data.length; i++) {
    const row = data[i] || [];
    const obj = {
      header_name: sanitize(row[0]) || '',
      company_name: companyName || null,
      add_col_name: addColName || null,
      user_id: userId
    };
    cols.forEach((c, idx) => {
      obj[c] = sanitize(row[idx]) || null;
    });
    records.push(obj);
  }

  if (records.length === 0) throw new Error('No data in Excel file');

  await db.SettingExcel.bulkCreate(records);
  return { count: records.length };
};
